<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Jobs\DownloadProductImage;

class ImportProductController extends Controller
{
    /**
     * Receives and saves product data sent by the external scraping service.
     */
    public function store(Request $request)
    {
        // --- 1. Validating Incoming Scraper Data ---
        $validator = Validator::make($request->all(), [
            'sku' => 'required|string|max:50', // The unique key for the product
            'name_fr' => 'required|string|max:255',
            'price' => 'required|numeric|min:0',
            'description_fr' => 'nullable|string',
            'image_urls' => 'required|array', // Array of image URLs
            'source_link' => 'required|url', // The original 1688 link
            // Optional, but highly recommended:
            'category_id' => 'nullable|integer',
            'current_stock' => 'nullable|integer', // New stock field
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $data = $validator->validated();

        // --- 2. Mapping Data to Columns in the Product.php Model ---
        $productData = [
            // Fields mapped from the Scraper
            'name' => $data['name_fr'], // Mapped to 'name' in the model
            'code' => $data['sku'], // Mapped to 'code' in the model
            'unit_price' => $data['price'],
            'details' => $data['description_fr'], // Mapped to 'details' in the model
            'external_link' => $data['source_link'], // Mapped to 'external_link' in the model
            'current_stock' => $data['current_stock'] ?? 0, // Use scraped stock or default to 0

            // Required fields in the 6Valley model with configurable default values
            'added_by' => 'admin', // Always imported by the admin via this system
            'product_type' => 'physical', // Assumes physical by default
            'status' => 0, // Set to Draft/Pending Initial Approval (0=Pending)
            'user_id' => env('IMPORT_DEFAULT_USER_ID', 1), // Configurable admin user ID
            'category_id' => $data['category_id'] ?? env('IMPORT_DEFAULT_CATEGORY_ID', 1), // Configurable category ID
            'minimum_order_qty' => 1,
            'category_ids' => json_encode([['id' => (string)($data['category_id'] ?? env('IMPORT_DEFAULT_CATEGORY_ID', 1)), 'position' => 1]]), // Structure required for 6Valley
        ];

        // --- 3. Creation or Update (Upsert) ---
        try {
            $product = Product::updateOrCreate(
                ['code' => $data['sku']], // Uses the code/SKU as a unique identifier
                $productData
            );

            // --- 4. Dispatch the Image Job (Asynchronous) ---
            // Downloads the images in the background to avoid blocking the Scraper.
            DownloadProductImage::dispatch($product, $data['image_urls']);

            return response()->json([
                'status' => 'success',
                'message' => 'Product imported/updated successfully.',
                'product_id' => $product->id
            ]);
        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'message' => 'Error saving product: ' . $e->getMessage()], 500);
        }
    }
}